
#include <p18f27j53.h>
#include <usart.h>
#include <stdio.h>
#include "fuses.h"
#include "inttypes.h"
#include "inlineasm.h"
#include "flash.h"
#include "ymodem.h"


const uint8_t far rom * first_log_page = (uint8_t far rom *)0x006800;
const uint8_t far rom * last_log_page =  (uint8_t far rom *)0x01F800;

uint8_t usb_power = 0;
unsigned seconds = 0;
uint8_t show_samples = 0;
uint16_t samples_todo = 0;
uint16_t prev_adc;
uint8_t *buffer_ptr;
uint8_t far rom * current_page = (uint8_t far rom *)0x006800; //first_log_page;
uint8_t far rom * next_page;

#pragma udata log_buf
uint8_t flash_buf[1024];
#pragma udata


void InterruptHandlerHighPriority();	// Forward declaration of handler
#pragma interrupt InterruptHandlerHighPriority
#pragma code high_vector = 0x000008		// Put the high priority interrupt handler at 0x0008
void int_high(void)
{										// Jump to the full handler somewhere else
	_asm
	goto InterruptHandlerHighPriority
	_endasm
}

void InterruptHandlerLowPriority();		// Forward declaration of handler
#pragma interruptlow InterruptHandlerLowPriority
#pragma code low_vector = 0x000018		// Put the low priority interrupt handler at 0x0018
void int_low(void)
{										// Jump to the full handler somewhere else
	_asm
	goto InterruptHandlerLowPriority
	_endasm
}
#pragma code

#define HLVDIF 2
#define HLVDIE 2

void Boost(void)
{
	HLVDCONbits.HLVDEN = 0;
	HLVDCON = 0xFF;
	PIR2 &= ~4;
	while(!HLVDCONbits.IRVST);
	usb_power = PIR2 & 4; //PIR2bits.HLVDIF;

	if(usb_power) return;
	//return;

	_asm

	movlb		0x0F

	bcf			HLVDCON, 4, _ACCESS		// Setup HLVD for max Vdd (3.5 V)
	movlw		0xFE					
	movwf		HLVDCON, _ACCESS
	bcf			PIR2, HLVDIF, _ACCESS

	bsf			CVRCON, 7, _BANKED		// Comparator voltage ref on
	bsf			CM1CON, 7, _ACCESS		// Comparator on

	bcf			LATA, 0, _ACCESS
	bcf			LATA, 1, _ACCESS
	bcf			LATA, 2, _ACCESS

	movf		TRISA, _W, _ACCESS
	andlw		0xF8
	movwf		PRODL, _ACCESS
	movf		TRISA, _W, _ACCESS
	iorlw		0x07
	movwf		PRODH, _ACCESS

	movf		PRODH, _W, _ACCESS
off3:
	movwf		TRISA, _ACCESS			// Switch off
	nop
off2:
	movwf		TRISA, _ACCESS			// Switch off
off1:
	movwf		TRISA, _ACCESS			// Switch off
off:
	movwf		TRISA, _ACCESS			// Switch off
	movff		PRODL, TRISA			// Switch on
	btfsc		PIR2, HLVDIF, _ACCESS
	bra			done
	btfss		CMSTAT, 0, _ACCESS
	bra			off3
	btfss		CMSTAT, 0, _ACCESS
	bra			off2
	btfss		CMSTAT, 0, _ACCESS
	bra			off2
	btfss		CMSTAT, 0, _ACCESS
	bra			off1
	btfss		CMSTAT, 0, _ACCESS
	bra			off
	btfss		CMSTAT, 0, _ACCESS
	bra			off
	btfss		CMSTAT, 0, _ACCESS
	bra			off
	btfss		CMSTAT, 0, _ACCESS
	bra			off
	bra			off						// Failed to saturate switch
done:
	movwf		TRISA, _ACCESS			// Switch off
	bcf			CM1CON, 7, _ACCESS		// Comparator off
	bcf			CVRCON, 7, _BANKED		// Comparator voltage ref off

	_endasm
}

void ShowTime(void)
{
	uint8_t yr,mo,da,wk,hr,mn,se;

	RTCCFGbits.RTCPTR0 = 1;
	RTCCFGbits.RTCPTR1 = 1;
	yr = RTCVALL;
	da = RTCVALH;
	da = RTCVALL;
	mo = RTCVALH;
	hr = RTCVALL;
	wk = RTCVALH;
	se = RTCVALL;
	mn = RTCVALH;

	seconds = (se & 0x0F) + ((se >> 4) * 10) + ((unsigned)(mn & 0x0F) * 60) + ((mn >> 4) * 600);

	printf((const rom char far *)"%02X.%02X.%02X %02X:%02x.%02x", yr, mo, da, hr, mn, se);
}

void LogSample(void)
{
	uint16_t adc;
	int16_t delta;
	uint8_t b;

	ADCON0bits.GO = 1;
	while(ADCON0bits.GO);
	adc = (unsigned)ADRESH << 8 | ADRESL;

	if(samples_todo == 0) {
		buffer_ptr = flash_buf;
		samples_todo = 1000;
		
		RTCCFGbits.RTCPTR0 = 1;
		RTCCFGbits.RTCPTR1 = 1;
		*buffer_ptr++ = RTCVALL;	// Year
		b = RTCVALH;				//
		b = RTCVALL;				// Day
		*buffer_ptr++ = RTCVALH;	// Month
		*buffer_ptr++ = b;			// Day
		b = RTCVALL;				// Hour
		*buffer_ptr++ = RTCVALH;	// Week
		*buffer_ptr++ = b;			// Hour
		b = RTCVALL;				// Second
		*buffer_ptr++ = RTCVALH;	// Minute
		*buffer_ptr++ = b;			// Second
		b = 16; do { *buffer_ptr++ = 0; } while(--b);
		*buffer_ptr++ = adc & 0xFF;
		*buffer_ptr++ = adc >> 8;
		prev_adc = adc;
		delta = 0;
	} else {
		delta = adc - prev_adc;
		if(delta < -127) {
			delta = -127;
		} else if(delta > 127) {
			delta = 127;
		}
		prev_adc += delta;
		*buffer_ptr++ = (uint8_t) delta;
	}

	--samples_todo;

	if(show_samples) {
		printf((const rom char far *)"%03i ", samples_todo);
		ShowTime();
		printf((const rom char far *)"  %u %u %i\r\n", adc, prev_adc, delta);
	}

	if(samples_todo == 0) {
		//printf((const rom char far *)"1000 values logged\r\n");
		next_page = (current_page == last_log_page) ? first_log_page : current_page + 1024;
		Boost();
		EraseFlashPage(next_page);
		Boost();
		WriteFlashPage(current_page, flash_buf);
		current_page = next_page;
	}
}

void InterruptHandlerHighPriority(void)
{
	if(PIR3bits.RTCCIF) {
		//printf((const rom char far *)"Tick\r\n");
		//ShowTime(); printf((const rom char far *)"\r\n");
		Boost();
		LogSample();
		PIR3bits.RTCCIF = 0;
	}
}

void InterruptHandlerLowPriority(void)
{
}


void Upload(void)
{
	const uint8_t rom far *p;
	const uint8_t rom far *end = last_log_page + 1024;

	show_samples = 0;

	BeginYModem();

	p = first_log_page;
	do {
		SendBlock(p);
		p += 128;
	} while(p != end);

	EndYModem();

	show_samples = 1;
}


void BoostTest(void)
{
	unsigned last_time = 0;

	do {
		Boost();						// --- Boost up to 3.5 volts

										// Show time
		Open1USART(USART_TX_INT_OFF & USART_RX_INT_OFF & USART_ASYNCH_MODE &\
			USART_EIGHT_BIT & USART_CONT_RX & USART_BRGH_HIGH,
			( 8000000 / 16 / 38400 ) - 1);
		ShowTime();
		//putc('\r', stdout); putc('\n', stdout);
		printf((const rom char far *)" %i\r\n\r\n", (seconds > last_time) ? seconds - last_time : seconds + 3600 - last_time);
		last_time = seconds;
		Close1USART();

										// --- Sleep until Vdd falls to 2.4 volts
		HLVDCONbits.HLVDEN = 0;			// Setup HLVD for min Vdd (2.5V)
		HLVDCON = 0x78;
		PIR2bits.LVDIF = 0;				// Reset HLVD flag
		PIE2bits.LVDIE = 1;				// Enable HLVD interrupt
		Sleep();						// Sleep until HLVD interrupt
		PIE2bits.LVDIE = 0;				// Disable HLVD interrupt

	} while(1);							// --- Do forever
}



void SetTime(void)
{
	const uint8_t year =   0x10;
	const uint8_t month =  0x11;
	const uint8_t day =    0x11;
	const uint8_t dow =    0x05;
	const uint8_t hour =   0x23;
	const uint8_t minute = 0x20;
	const uint8_t second = 0x00;

	RTCCFGbits.RTCPTR0 = 1;
	RTCCFGbits.RTCPTR1 = 1;
	RTCVALL = year;		// Year
	RTCVALH = 0;		//
	RTCVALL = day;		// Day
	RTCVALH = month;	// Month
	RTCVALL = hour;		// Hour
	RTCVALH == dow;		// Week
	RTCVALL = second;	// Second
	RTCVALH = minute;	// Minute
	
}

void main(void)
{
	uint8_t b;
	uint16_t x;

	OSCCONbits.IRCF = 7;				// Run at 8 MHz


	ANCON0 = 0xFF ^ 0x1F;				// AN0, AN1, AN2, AN3 and AN4 (RA5) as analog
	ANCON1 = 0xFF ^ 0x01;				// AN8 (RB2) as analog

	TRISA = 0xEF;
	LATA = 0;

	TRISB = 0x04;
	LATB = 0;

	TRISC = 0x80;
	LATC = 0x40;

	HLVDCONbits.HLVDEN = 1;				// Enable HLVD circuit

	WDTCONbits.REGSLP = 1;				// Low power sleep mode

	ADCON0bits.VCFG0 = 1;				// Vref = AN3
	ADCON1bits.ADFM = 1;				// Right justified result
	ADCON1bits.ADCS = 1;				// Fosc/8 conversion clock
	ADCON0bits.CHS = 8;					// Channel 8
	ADCON0bits.ADON = 1;				// Turn on ADC

	CVRCONbits.CVR = 2;
	CVRCONbits.CVRR = 1;
	CVRCONbits.CVRSS = 0;
	//CVRCONbits.CVREN = 1;

	CM1CONbits.CCH = 2;
	//CM1CONbits.CON = 1;

	T1CONbits.T1OSCEN = 1;				// Enable 32.768 kHz osc

	INTCONbits.GIE = 0;					// Enable RTC writes
	EECON2 = 0x55;
	EECON2 = 0xAA;
	RTCCFGbits.RTCWREN = 1;
	RTCCFGbits.RTCEN = 1;				// Enable RTC

	ALRMCFGbits.AMASK0 = 1;				// Alarm every second
	ALRMCFGbits.AMASK1 = 0;
	ALRMCFGbits.AMASK2 = 0;
	ALRMCFGbits.AMASK3 = 0;
	ALRMCFGbits.CHIME = 1;				// Repeat forever
	ALRMCFGbits.ALRMEN = 1;				// Enable alarm

	PIR3bits.RTCCIF = 0;
	PIE3bits.RTCCIE = 1;
	INTCONbits.PEIE = 1;
	INTCONbits.GIE = 1;
	

	Open1USART(USART_TX_INT_OFF & USART_RX_INT_OFF & USART_ASYNCH_MODE &\
		USART_EIGHT_BIT & USART_CONT_RX & USART_BRGH_HIGH,
		( 8000000 / 16 / 38400 ) - 1);

	printf((const rom char far *)"Logger Up\r\n");

	//SetTime();

	//BoostTest();

	show_samples = 1;
	do {
		Boost();
		if(usb_power) {
			if(DataRdy1USART()) {
				b = Read1USART();
				if(b == 'C') {
					Upload();
				} else {
					printf((const rom char far *)"WTF? %c\r\n", b);
				}
			}
		} else {
			while(Busy1USART());
			Boost();
			Sleep();
			//printf((const rom char far *)"Wakeup\r\n");
		}
	} while(1);

	while(1);
}
