

; IR Widget firmware for MiniPOV3
;
; Copyright (C) 2008 Kevin Timmerman
;
; irwidget [@t] compendiumarcana [d0t] com
; http://www.compendiumarcana.com/irwidget
;
;
; This program is free software: you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation, either version 3 of the License, or
; (at your option) any later version.
;
; This program is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with this program.  If not, see <http://www.gnu.org/licenses/>.
;
;
;
; ---> Set ATtiny2313 fuse for external osc <---
;
; avrdude -p t2313 -c dasa -P COMx -U lfuse:w:0xCE:m
;
; ---> Write firmware <---
;
; avrdude -p t2313 -c dasa -P COMx -U flash:w:povcap.hex
;


	;
	; AVR	IR LED
	;------------------
	; 7 D3	Anode   + Long Lead
	; 8 D4	Cathode - Short Lead (Flat)
	;
	; AVR	QSE157
	;------------------
	; 7 D3	Ground
	; 8 D4	Out
	; 9 D5	Power

	; DE9			tiny2323
	;-------------------------
	; 1 CD		<-
	; 2 RxD		<-	3  D1 TxD
	; 3 TxD		->	17 B5 DI
	; 4	DTR		->	19 B7 CLK
	; 5 Gnd		--	10    Gnd
	; 6 DSR		<-
	; 7	RTS		->	1  A2 Reset
	; 8	CTS		<-	18 B6 DO
	; 9 RI		<-
	;
	;

	;.DEVICE	ATtiny2313							; This is in the include file
													;
	.INCLUDE	"tn2313def.inc"						; Register/Bit Definitions for the ATtiny2313
													;
													;
													; - Register usage
	.DEF	temp        = r16						;
	.DEF	temp2       = r17						;
	.DEF	delay_cnt   = r18						;
	.DEF	tx_data     = r19						;
	.DEF	bit_count   = r20						;
	.DEF	prev_count  = r21						;
	.DEF	period_mask = r21						;
	.DEF	time_save_l = r22						;
	.DEF	time_save_h = r23						;
													;
													; - I/O Port usage
													;
	.EQU	tx_port         = PORTD					; Serial data to host
	.EQU	tx_bit          = 1						;
	.EQU	mode1_port      = PINB					; Mode select
	.EQU	mode1_bit       = 7						;
	.EQU	mode2_port      = PINB					;
	.EQU	mode2_bit       = 5						;
	.EQU	status_led_port = PORTB					; Status LED
	.EQU	status_led_bit  = 0						;
	.EQU	ir_demod_port   = PIND					; IR Demodulator input
	.EQU	ir_demod_pin    = 2						;
													;
													;
													; - Interrupt Vectors
													;
	rjmp	init									;  1 0000 Reset
	rjmp	bad_int									;  2 0001 INT0
	rjmp	bad_int									;  3 0002 INT1
	rjmp	bad_int									;  4 0003 T1 Capture
	rjmp	bad_int									;  5 0004 T1 Compare A
	rjmp	bad_int									;  6 0005 T1 Overflow
	rjmp	bad_int									;  7 0006 T0 Overflow
	rjmp	bad_int									;  8 0007 UART Rx
	rjmp	bad_int									;  9 0008 UART Tx Empty
	rjmp	bad_int									; 10 0009 UART Tx Complete
	rjmp	bad_int									; 11 000A Analog Comparator
	rjmp	bad_int									; 12 000B Pin Change
	rjmp	bad_int									; 13 000C T1 Compare B
	rjmp	bad_int									; 14 000D T0 Compare A
	rjmp	bad_int									; 15 000E T0 Compare B
	rjmp	bad_int									; 16 000F USI Start
	rjmp	bad_int									; 17 0010 USI Overflow
	rjmp	bad_int									; 18 0011 EEPROM
	rjmp	bad_int									; 19 0012 Watchdog
													;
bad_int:											; Unused interrupts
	rjmp	PC										;
													;
													;
init:												; --- Initialize
	ldi		temp,0b01011111							; Mode selects are inputs, all other outputs (LEDs)
	out		DDRB,temp								;
	ldi		temp,0b00000000							; Turn off all LEDs
	out		PORTB,temp								;
													;
	ldi		temp,0b00101010							; Tx data and IR pulse detector ground/power are outputs,
	out		DDRD,temp								;  all others inputs
	ldi		temp,0b00110000							; Enable pullup for IR pulse detector input, power for QSE157
	out		PORTD,temp								;
													;
	ldi		temp,0									; Configure Timer/Counter 0 for external input on D4 (pin 8)
	out		TCCR0A,temp								;
	ldi		temp,(1<<CS02)|(1<<CS01)				;
	out		TCCR0B,temp								;
													;
	ldi		temp,0									; Configure Timer/Counter 1 for internal clock with /64 prescale
	out		TCCR1A,temp								;  (8 uS period per increment, 524 mS rollover)
	out		TCCR1C,temp								;
	ldi		temp,(1<<CS11)|(1<<CS10)				;
	out		TCCR1b,temp								;
													;
	ldi		temp,0xDF								; Set stack pointer to top of RAM
	out		SPL,temp								;
													;
	sbic	mode1_port,mode1_bit					; Test mode select input
	rjmp	time_mode								; Goto time mode
	;rjmp	count_mode								; Goto count mode
													;
													;
													;
count_mode:											; --- Count mode
													;  Count the number of IR pulses within a 100 uS period
													;   and send to host at 115200,8,N,1
													;
count_wait:											; Wait for signal from host to proceed
	sbis	mode1_port,mode1_bit					;
	rjmp	count_wait								;
													;	
	in		tx_data,TCNT0							; Get current count, save for comparison
count_first:										; Wait for first pulse
	in		temp,TCNT0								; Get current count
	cp		tx_data,temp							; Compare to previous
	breq	count_first								; Keep waiting if same
	sbi		status_led_port,status_led_bit			; Turn on status LED
	rjmp	count_tx								; Send initial count...
count_loop:											;
	in		tx_data,TCNT0							; Get counter 1 count
													;
	cp		tx_data,prev_count						; Compare to previous count
	in		temp,SREG								;
	sbrc	temp,SREG_Z								; Turn off status LED if count is the same (Zero bit set)
	cbi		status_led_port,status_led_bit			;
	sbrs	temp,SREG_Z								; Turn on status LED if count has changed (Zero bit clear)
	sbi		status_led_port,status_led_bit			;
													;
count_tx:											;
	mov		prev_count,tx_data						; Save current count for next iteration
													;
	rcall	tx_115									; Send count to host
													;
	ldi		delay_cnt,6								; Make this loop exactly 100 uS long
	rcall	delay									;
													;
	sbic	mode1_port,mode1_bit					; Host wants more?
	rjmp	count_loop								; Yes, do it all again...
	rjmp	count_wait								; No, wait for signal to start over...
													;
													;
													;
time_mode:											; --- Time mode
													;  Measure the on and off times of the IR demodulator output
													;   and send 15 bit duration values to host at 115200,8,N,1
wait_low:											;
	sbic	ir_demod_port,ir_demod_pin				; Wait for the IR demod output to go low
	rjmp	wait_low								;  (low == IR present)
	ldi		period_mask,0x00						; Tell the host that this is the off period duration
	rcall	report_time								; Send duration to host
	sbi		status_led_port,status_led_bit			; Turn on status LED
													;
wait_high:											;
	sbis	ir_demod_port,ir_demod_pin				; Wait for the IR demod output to go high
	rjmp	wait_high								;  (high == IR absent)
	ldi		period_mask,0x80						; Tell the host that this is the on period duration
	sbic	mode1_port,mode1_bit					; Check the mode select, skip report if low
	rcall	report_time								; Send duration to host
	cbi		status_led_port,status_led_bit			; Turn off status LED
													;
	rjmp	wait_low								; Loop...
													;
													;
report_time:										; --- Send period duration to host
	in		temp,TCNT1L								; Get current time
	in		temp2,TCNT1H							;
													;
	sub		temp,time_save_l						; Calculate duration (elapsed time)
	sbc		temp2,time_save_h						;  (subtract previous time from current time)
													;
	add		time_save_l,temp						; Update saved time for next iteration
	adc		time_save_h,temp2						;  (add elapsed time to previous time)
													;
	lsr		temp2									; Divide by two (make 16 uS resolution, make room for mask)
	ror		temp									;
													;
	or		temp2,period_mask						; Flag on/off period
													;
	mov		tx_data,temp							; Send LSB to host
	rcall	tx_115									;
	mov		tx_data,temp2							; Send MSB to host
	rcall	tx_115									;
													;
	ret												; Return
													;
													;
													;
													; --- 115200,8,N,1 async serial transmit
tx_115:												;
	sbi		tx_port,tx_bit							; Start bit
	ldi		bit_count,10							; 10 bits to send (start, 8 data, stop)
	ldi		delay_cnt,46							; Wait for bit duration
	rcall	delay									;
tx_loop:											; - Do a bit
	sec												; Shift in stop bit
	ror		tx_data									; Shift out next bit
	brcs	tx_one									; One...
	nop												; Zero
	sbi		tx_port,tx_bit							; Signal zero bit (data is inverted!)
	rjmp	tx_next									;
tx_one:												;
	cbi		tx_port,tx_bit							; Signal one bit (data is inverted!)
	rjmp	tx_next									;
tx_next:											;
	ldi		delay_cnt,40							; Wait for bit duration
	sbrc	bit_count,0								; Keep timing perfect (69.444 instruction cycles per bit)
	ldi		delay_cnt,41							;  (alternate between 69 and 70 cycles)
	rcall	delay									;
	dec		bit_count								; Decrement bit count
	brne	tx_loop									; Next bit...
	ret												; Return
													;
													;
													; --- Delay with one instruction cycle granularity
delay:												;
	sbrc	delay_cnt,0								; Test bit 0
	rjmp	PC+1									; + 1 cycle if bit 0 is set
	sbrs	delay_cnt,1								; Test bit 1
	rjmp	PC+3									;
	rjmp	PC+1									; + 2 cycles if bit 1 is set
	nop												;
	cbr		delay_cnt,3								; Mask off bits 0 and 1, test for zero
delayl4:											;
	breq	delayex									; Exit if delay count is zero
	subi	delay_cnt,4								; Decrement delay count by 4 every 4 instruction cycles
	rjmp	delayl4									;
delayex:											;
	ret												; Return
													;
													;
	.EXIT											; End of source file
													;

