;
;    JP1 EEPROM Programming Adapter Firmware
;
;    Copyright (C) 2010 Kevin Timmerman
;    jp1epa [@t] compendiumarcana [d0t] com
;    http://www.compendiumarcana.com/jp1epa
;
;    This program is free software; you can redistribute it and/or modify
;    it under the terms of the GNU General Public License as published by
;    the Free Software Foundation; either version 3 of the License, or
;    (at your option) any later version.
;
;    This program is distributed in the hope that it will be useful,
;    but WITHOUT ANY WARRANTY; without even the implied warranty of
;    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;    GNU General Public License for more details.
;
;    You should have received a copy of the GNU General Public License
;    along with this program.  If not, see <http://www.gnu.org/licenses/>.
;
;


#include <p12hv609.inc>

	__config _BOR_ON & _IOSCFS_8MHZ & _CP_OFF & _MCLRE_OFF & _PWRTE_ON & _WDT_OFF & _INTOSCIO

	radix dec

	cblock 0x40
	flags
	temp1
	ee_size
	ee_fill
	jp1_cmd
	jp1_len
	jp1_chk
	addrh
	addrl
	asio_bit_time
	asio_data
	asio_flags
	iic_dly
	iic_data_rx
	iic_data_tx
	iic_ack_rx
	iic_ack_tx
	iic_bit_count
	data_buf
	endc

													; - Flags
fTwoByteAddress		equ	0							;
fFillTestPattern	equ	1							;
fShowAckWait		equ 2							;
													;
													;- I/O
asio_out_port	equ		GPIO						;
asio_in_port	equ		GPIO						;
pATN			equ		GPIO						;
pSCL			equ		GPIO						;
pSDA			equ		GPIO						;
pReset			equ		GPIO						;
													;
													; Pin 1 - Power
bSDA			equ		5							; Pin 2 - EEPROM SDA
bSCL			equ		4							; Pin 3 - EEPROM SCL	
bATN			equ		3							; Pin 4 - ATN
bReset			equ		2							; Pin 5 - Remote Reset
asio_out_bit	equ		1							; Pin 6 - Serial Out	
asio_in_bit		equ		0							; Pin 7 - Serial In
													; Pin 8 - Gnd
													;



													;
													;
Start												; - Setup I/O
	clrf		GPIO								;
	movlw		(1<<bATN) | (1<<asio_in_bit) | (1<<bSDA) ; | (1<<bReset)
	bsf			STATUS, RP0							;
	movwf		GPIO								;
	clrf		ANSEL								; Turn off analog stuff
	bcf			STATUS, RP0							;
													;
	movlw		26 * 2 - 24							; 26 uS == 38,461
	movwf		asio_bit_time						;
													;
													;
	clrf		flags								; Clear all flags
													;
	goto		JP1E_C16							; Setup defaults for 24C16, and wait for command...
													;
													;
													;
													;
													;
cmd_lookup											; --- Command Table ---
	bcf			PCLATH, 0							; Must be in 0x0000 page
	movf		temp1, W							;
	addwf		PCL, F								;
													; - Standard commands
	retlw		'R'									; R - Read
	goto		JP1E_R								;
	retlw		'S'									; S - Write
	goto		JP1E_S								;
	retlw		'C'									; C - Erase
	goto		JP1E_C								;
	retlw		'E'									; E - Ping
	goto		JP1E_E								;
	retlw		'I'									; I - Identity
	goto		JP1E_I								;
	retlw		'V'									; V - Version
	goto		JP1E_V								;
													; - Extended commands
	retlw		'd'									; d - Display EEPROM contents
	goto		JP1E_d								;
	retlw		'f'									; f - Fill EEPROM with 0xFF
	goto		JP1E_f								;
	retlw		'z'									; z - Fill EEPROM with 0x00
	goto		JP1E_z								;
	retlw		't'									; t - Fill EEPROM with test pattern
	goto		JP1E_t								;
	retlw		'1'									; 1 - One byte addressing
	goto		JP1E_1								;
	retlw		'2'									; 2 - Two byte addressing
	goto		JP1E_2								;
	retlw		'4'									; 4 - 24C04 - 512 byte
	goto		JP1E_C04							;
	retlw		'8'									; 8 - 24C08 - 1K byte
	goto		JP1E_C08							;
	retlw		'6'									; 6 - 24C16 - 2K byte
	goto		JP1E_C16							;
	retlw		'3'									; 3 - 24C32 - 4K byte
	goto		JP1E_C32							;
	retlw		'x'									; x - Exit
	goto		JP1E_x								;
	retlw		'?'									; ? - Product Info
	goto		JP1E_info							;
													;
	retlw		0									; End of command list
													;
													;
send_msg											; - Send text to PC
	movwf		temp1								; Save index
msg_loop											;
	movf		temp1, W							; Get index
	call		get_msg_char						; Get char
	xorlw		0									; End of string?
	btfsc		STATUS, Z							;
	return											; Yes, exit...
	xorwf		jp1_chk, F							; Update checksum
	call		SerTx								; Send to PC
	incf		temp1, F							; Increment index
	goto		msg_loop							; Next char...
													;
	org			0x0100								; Text must be in 0x0100 page!
													;
get_msg_char										;
	bsf			PCLATH, 0							;
	addwf		PCL, F								;
msg_area											;
msg_eeprom											;
	dt			"eeprom"							;
	dt			0									;
msg_info											;
	dt			"JP1 EEPROM Programming Adapter\r\n"
	dt			"Copyright 2010 Kevin Timmerman\r\n"
	dt			"Rev 003 Config "					;
	dt			0									;
msg_help											;
	dt			"d Display EEPROM contents\r\n"		;
	dt			"f Fill with FF\r\n"				;
	dt			"z Fill with 00\r\n"				;
	dt			"t Fill with test pattern\r\n"		;
	dt			"4 24C04\r\n"						;
	dt			"8 24C08\r\n"						;
	dt			"6 24C16\r\n"						;
	dt			"3 24C32"							;
	dt			0									;
msg_fill											;
	dt			"Fill EEPROM with "					;
	dt			0									;
msg_test											;
	dt			"test pattern"						;
	dt			0									;
msg_fill_abort										;
	dt			"Fill Canceled"						;
	dt			0									;
													;
													;
													;
													;
JP1E_ACK											; --- Send ACK to PC
	movlw		6									; ACK
	call		SerTx								;
	;goto		JP1E_next							; Done...
													;
JP1E_next											; --- Get command from PC and execute it
	call		SerRx								; Get command from PC
	movwf		jp1_cmd								; Save command
	movwf		jp1_chk								; Initialize checksum
													;
	bcf			flags, fShowAckWait					; Default to not showing ACK wait
													;
	clrf		temp1								; Clear command lookup index
cmd_loop											;
	call		cmd_lookup							; Get command char
	xorlw		0									; End of table?
	btfsc		STATUS, Z							;
	goto		JP1E_next							; Yes...
	incf		temp1, F							; Increment index to point to handler
	xorwf		jp1_cmd, W							; Command char match?
	btfsc		STATUS, Z							;
	goto		cmd_lookup							; Yes, jump to handler...
	incf		temp1, F							; Increment index to point to next command char
	goto		cmd_loop							; Try next...
													;
													;
JP1E_E												; --- Ping
													; -> E
													; <- <ACK>
	goto		JP1E_ACK							;
													;
JP1E_I												; --- Identity
													; -> I
													; <- [data] <chk>
													;
	clrf		jp1_chk								; Init checksum
	movlw		msg_eeprom - msg_area				; "eeprom"
	call		send_msg							;
	movf		jp1_chk, W							; <chk>
	call		SerTx								;
													;
	goto		JP1E_next							; Done...
													;
													;
JP1E_V												; --- Version
													; -> V
													; <- [data] <chk>
													;
													; Flash size is in units of 128 bytes
	bcf			STATUS, C							;
	rlf			ee_size, W							; Get max MSB of EE address * 2
	movwf		jp1_chk								;
	call		SerTx								;
	movlw		0									; EEPROM base address MSB
	call		SerTx								;
	movlw		0									; EEPROM base address LSB
	call		SerTx								;
	movf		jp1_chk, W							;
	call		SerTx								;
													;
	goto		JP1E_next							; Done...
													;
JP1E_R												; --- Read
													; -> R <addrh> <addrl> <len> <chk>
													; <- r [data] <chk>
													;
	call		get_addr_len						; Get address and length
	call		SerRx								; Get checksum
	xorwf		jp1_chk, F							; Update checksum
	btfss		STATUS, Z							;
	goto		JP1E_next							; Ignore if bad checksum...
	movlw		'r'									; Response begins with 'r'
	movwf		jp1_chk								;
	call		SerTx								;
													;
	call		ee_addr								; Setup address
	call		iic_stop							; End write operation
													;
	call		iic_start							; Begin read operation
	rlf			addrh, W							; Get high address shifted left by one bit
	andlw		0x0E								; Mask off IIC address and R/W bits
	btfsc		flags, fTwoByteAddress				; Zero for two byte addressing
	movlw		0									;
	iorlw		0xA1								; OR with EEPROM IIC address and read command (bit 0 == 1)
	call		iic_byte							;
													;
	decf		jp1_len, F							; Do N-1 in loop, last byte has no ACK
	btfsc		STATUS, Z							;
	goto		jp1rlast							; Only one byte...
	clrf		iic_ack_tx							; ACK each byte
jp1rloop											; - Read loop
	movlw		0xFF								; Allow slave to drive SDA
	call		iic_byte							; Get a byte
	movf		iic_data_rx, W						;
	xorwf		jp1_chk, F							; Update checksum
	call		SerTx								; Send to PC
	decfsz		jp1_len, F							; Dec byte count
	goto		jp1rloop							; Loop if more..
													;
jp1rlast											; - Read last byte
	bsf			iic_ack_tx, 0						; Don't ACK last byte
	movlw		0xFF								; Allow slave to drive SDA
	call		iic_byte							; Get a byte
	movf		iic_data_rx, W						;
	xorwf		jp1_chk, F							; Update checksum
	call		SerTx								; Send to PC
													;
	call		iic_stop							; End IIC operaion
													;
	movf		jp1_chk, W							; Get checksum
	call		SerTx								; Send to PC
	goto		JP1E_next							; Done...
													;
JP1E_S												; --- Write
													; -> S <addrh> <addrl> <len> [data] <chk>
													; <- <ACK>
													;
	call		get_addr_len						; Get address and length
	movlw		data_buf							; Setup buffer pointer
	movwf		FSR									;
	movf		jp1_len, W							; Make a copy of length
	movwf		temp1								;
next_S_byte											;
	call		SerRx								; Get data
	movwf		INDF								; Save in buffer
	incf		FSR, F								; Increment buffer pointer
	xorwf		jp1_chk, F							; Update checksum
	decfsz		temp1, F							; Decrement length, check if zero
	goto		next_S_byte							;
	call		SerRx								; Get checksum
	xorwf		jp1_chk, F							; Update checksum
	btfss		STATUS, Z							;
	goto		JP1E_next							; Ingore if bad checksum...
													;
													; - Write to EEPROM
	movlw		data_buf							; Reset buffer pointer
	movwf		FSR									;
next_ee_page										;
	call		ee_addr								; Wait for EEPROM to be ready, send address
													;
next_ee_byte										;
	movf		INDF, W								; Get a byte from buffer
	incf		FSR, F								; Increment buffer pointer
	call		iic_byte							; Write to EEPROM
	incf		addrl, F							; Increment address
	btfsc		STATUS, Z							;
	incf		addrh, F							;
													;
	decf		jp1_len, F							; Decrement length
	btfsc		STATUS, Z							;
	goto		S_done								; Exit if zero...
	movf		addrl, W							; Check if end of EEPROM page
	andlw		0x1F								; Assume 32 byte page
	btfss		flags, fTwoByteAddress				; Skip next if two byte addressing
	andlw		0x0F								; 16 byte page
	btfss		STATUS, Z							;
	goto		next_ee_byte						; Not page boundry, do next byte...
	call		iic_stop							; Write page
	goto		next_ee_page						; Next page...
													;
S_done												;
	call		iic_stop							; Begin write of last page
	call		ee_wait								; Wait for write to complete
	call		iic_stop							; End IIC operation
													;
	goto		JP1E_ACK							; ACK the write...
													;
													;
JP1E_C												; --- Erase
													; -> C <addrh> <addrl> <counth> <countl> <chk>
													; <- <ACK>
													;
	call		get_addr_len						;
	call		SerRx								; Get countl
	xorwf		jp1_chk, F							; Update checksum
	call		SerRx								; Get checksum
	xorwf		jp1_chk, F							; Update checksum
	btfss		STATUS, Z							;
	goto		JP1E_next							; Ingore bad checksum...
	goto		JP1E_ACK							; ACK good checksum...
													;
													;
JP1E_d												; --- d - Display EEPROM contents
	call		SerCRLF								; Send CR/LF to PC
													;
	bsf			flags, fShowAckWait					; Show ACK wait
													;
	call		iic_start							; Begin IIC operation
													;
	bsf			iic_ack_tx, 0						; Slave will ACK
	movlw		0xA0								; Send write command to eeprom (write address)
	call		ee_debug							; Show results of IIC activity
													;
	movlw		0x00								; Send address MSB for large EEPROMs
	btfsc		flags, fTwoByteAddress				;
	call		ee_debug							;
													;
	movlw		0x00								; Send address LSB
	call		ee_debug							;
													;
	call		iic_stop							; End of IIC operation
													;
	call		iic_start							; Begin IIC operation
													;
	movlw		0xA1								; Send read command to EEPROM
	call		ee_debug							;
													;
	clrf		addrh								; Clear address
	clrf		addrl								;
													;
	clrf		iic_ack_tx							; Host will ACK
idrl												;
	movf		addrl, W							; Check if LSN of address == 0
	andlw		0x0F								;
	btfss		STATUS, Z							;
	goto		notbol								; No...
													; - Send address to PC
	call		SerCRLF								; CR/LF
	movf		addrh, W							;
	call		tx_hex_byte							; Address
	movf		addrl, W							;
	call		tx_hex_byte							;
	movlw		':'									; ':'
	call		SerTx								;
													;
notbol												;
	movlw		0xFF								; Read a byte from EEPROM
	call		iic_byte							;
													;
	movlw		' '									; <space>
	call		SerTx								;
	movf		iic_data_rx, W						; Show byte in hex
	call		tx_hex_byte							;
													;
	incf		addrl, F							; Increment address
	btfsc		STATUS, Z							;
	incf		addrh, F							;
													;
	incf		addrl, W							; Check if end of EEPROM
	btfss		STATUS, Z							;
	goto		idrl								; LSB != 0xFF
	decf		ee_size, W							;
	xorwf		addrh, W							;
	btfss		STATUS, Z							;
	goto		idrl								; MSB != EEPROM size MSB - 1
													;
	bsf			iic_ack_tx, 0						; No ACK for last byte
	movlw		0xFF								;
	call		iic_byte							; Read last byte
													;
	movlw		' '									; <space>
	call		SerTx								;
	movf		iic_data_rx, W						; Show byte in hex
	call		tx_hex_byte							;
													;
	call		iic_stop							; End IIC operation
													;
	call		SerCRLF								; CR/LF
													;
	goto		JP1E_next							; Done...
													;
													;
													;
													;
JP1E_1												; 1 - One byte addressing
	bcf			flags, fTwoByteAddress				;
	goto		JP1E_ACK							;
													;
JP1E_2												; 2 - Two byte addressing
	bsf			flags, fTwoByteAddress				;
	goto		JP1E_ACK							;
													;
JP1E_C04											; 4 - 24C04 - 512 byte EEPROM
	movlw		0x02								;
set_ee_size											;
	movwf		ee_size								;
	goto		JP1E_1								; Set one byte addressing...
													;
JP1E_C08											; 8 - 24C08 - 1K byte EEPROM
	movlw		0x04								;
	goto		set_ee_size							;
													;
JP1E_C16											; 6 - 24C16 - 2K byte EEPROM
	movlw		0x08								;
	goto		set_ee_size							;
													;
JP1E_C32											; 3 - 24C32 - 4K byte EEPROM
	movlw		0x10								;
	movwf		ee_size								;
	goto		JP1E_2								; Set two byte addressing...
													;
JP1E_f												; f - Fill EEPROM with 0xFF
	movlw		0xFF								;
	movwf		ee_fill								;
	bcf			flags, fFillTestPattern				; Clear test pattern flag
	goto		JP1E_fill							;
													;
JP1E_z												; z - Fill EEPROM with 0x00
	clrf		ee_fill								;
	bcf			flags, fFillTestPattern				; Clear test pattern flag
	goto		JP1E_fill							;
													;
JP1E_t												; t - Fill EEPROM with test pattern
	bsf			flags, fFillTestPattern				; Set test pattern flag
	goto		JP1E_fill							;
													;
JP1E_x												; x - Release remote reset
	goto		JP1E_next							;
													;
JP1E_info											; ? - Product information
	call		SerCRLF								;
	movlw		msg_info - msg_area					; Copyright and revision
	call		send_msg							;
	movf		ee_size, W							;
	call		tx_hex_byte							; EE size
	movlw		' '									;
	call		SerTx								;
	movlw		'2'									; EE adress bytes
	btfss		flags, fTwoByteAddress				;
	movlw		'1'									;
	call		SerTx								;
	call		SerCRLF								;
	call		SerCRLF								;
	movlw		msg_help - msg_area					; Help
	call		send_msg							;
	call		SerCRLF								;
	goto		JP1E_next							;
													;
JP1E_fill											; --- Fill EEPROM with ee_fill
	movlw		msg_fill - msg_area					; Prompt for fill comfirmation
	call		send_msg							;
	movlw		msg_test - msg_area					;
	btfsc		flags, fFillTestPattern				; "test pattern"
	call		send_msg							;
	movf		ee_fill, W							;
	btfss		flags, fFillTestPattern				;
	call		tx_hex_byte							; hex fill byte
	movlw		' '									;
	call		SerTx								;
	movlw		'?'									; '?'
	call		SerTx								;
	call		SerCRLF								;
													;
	call		SerRx								; Must confirm with 'Y' to proceed
	xorlw		'Y'									;
	btfss		STATUS, Z							;
	goto		fill_abort							; Abort if not 'Y'...
													;
													;
	clrf		addrl								; Clear address
	clrf		addrh								;
													;
	bsf			flags, fShowAckWait					; Show ACK wait
													;
eefill_page											; - Fill page
	call		SerCRLF								; CR/LF
	movf		addrh, W							;
	call		tx_hex_byte							; Show address in hex
	movf		addrl, W							;
	call		tx_hex_byte							;
													;
	call		ee_addr								; Setup eeprom address
													;
eefill_byte											; - Write byte
	movf		addrl, W							; Test pattern data == addr MSB + addr LSB
	addwf		addrh, W							;
	btfss		flags, fFillTestPattern				;
	movf		ee_fill, W							; Get fill data
	call		iic_byte							; Write it to eeprom
													;
	incf		addrl, F							; Increment address
	btfsc		STATUS, Z							;
	incf		addrh, F							;
													;
	movf		addrl, W							; Check if end of EEPROM page
	andlw		0x1F								; Assume 32 byte page
	btfss		flags, fTwoByteAddress				; Skip next if two byte addressing
	andlw		0x0F								; 16 byte page
	btfss		STATUS, Z							;
	goto		eefill_byte							; Not page boundry, do next byte...
													;
	call		iic_stop							; Stop, write page
													;
	movf		ee_size, W							; Get EE size
	xorwf		addrh, W							; Reached end?
	btfss		STATUS, Z							;
	goto		eefill_page							; No, do next page...
													;
	call		ee_wait								; Wait for last page write to complete
	call		iic_stop							; End IIC operation
	call		SerCRLF								;
	goto		JP1E_next							; Done...
													;
													;
fill_abort											; - Abort fill
	movlw		msg_fill_abort - msg_area			; Abort message
	call		send_msg							;
	call		SerCRLF								; CR LF
	goto		JP1E_next							; Done...
													;
													;
ee_debug											; --- Show IIC data byte and ACK bit
	call		iic_byte							; Do IIC byte xfer
	movf		iic_data_rx, W						; Get rx IIC data
	call		tx_hex_byte							; Show byte in hex
	movlw		' '									; <space>
	call		SerTx								;
	movf		iic_ack_rx, W						; Get rx IIC ACK
	call		tx_hex_nibble						; Show bit
	movlw		' '									; <space>
	goto		SerTx								;
													;
													;
get_addr_len										; --- Get address and length from host, update checksum
	call		SerRx								; Get address high byte
	movwf		addrh								;
	xorwf		jp1_chk, F							;
	call		SerRx								; Get address low byte
	movwf		addrl								;
	xorwf		jp1_chk, F							;
	call		SerRx								; Get length
	movwf		jp1_len								;
	xorwf		jp1_chk, F							;
	return											;
													;
ee_addr												; --- Wait for EEPROM to be ready and write whole address
	call		ee_wait								; Wait for ready, send write command
	movf		addrh, W							; Get high address
	btfsc		flags, fTwoByteAddress				; Skip if one byte addressing
	call		iic_byte							; Send high address
	movf		addrl, W							; Get low address
	goto		iic_byte							; Send low address and return
													;
													;
ee_wait												; --- Wait for EEPROM to be ready, send write command
	bsf			iic_ack_tx, 0						; Slave will ACK
waitack												;
	call		iic_start							; Begin IIC operation
	rlf			addrh, W							; Get high address shifted left by one bit
	andlw		0x0E								; Mask off IIC address and R/W bits
	btfsc		flags, fTwoByteAddress				; Zero for two byte addressing
	movlw		0									;
	iorlw		0xA0								; OR with EEPROM IIC address and write command (bit 0 == 0)
	call		iic_byte							; Sent to EEPROM
	btfss		iic_ack_rx, 0						; Test rx ACK
	return											; Return if EEPROM ACK'd
	movlw		'.'									; Send '.' to PC
	btfsc		flags, fShowAckWait					; Only if enabled
	call		SerTx								;
	goto		waitack								; Continue to wait
													;
													;
													;
													; ******************** IIC I/O *******************
													;
iic_start											; --- IIC Start - SDA fall while SCL high
													;
	call		iic_delay							; Delay
													;
	bsf			pSCL, bSCL							; SCL high
													;
	call		iic_delay							; Delay
													;
	bcf			pSDA, bSDA							; SDA low
	bsf			STATUS, RP0							;
	bcf			pSDA, bSDA							;
	bcf			STATUS, RP0							;
													;
	call		iic_delay							; Delay
													;
	bcf			pSCL, bSCL							; SCL low
													;
	return											;
													;
													;
iic_stop											; --- IIC Stop - SDA rise while SCL high
													;
	call		iic_delay							; Delay
													;
	bcf			pSDA, bSDA							; SDA low
	bsf			STATUS, RP0							;
	bcf			pSDA, bSDA							;
	bcf			STATUS, RP0							;
													;
	call		iic_delay							; Delay
													;
	bsf			pSCL, bSCL							; SCL high
													;
	call		iic_delay							; Delay
													;
	bsf			STATUS, RP0							; SDA high
	bsf			pSDA, bSDA							;
	bcf			STATUS, RP0							;
	return											;
													;
iic_byte											; --- Exchange IIC byte and ACK
													;
	movwf		iic_data_tx							; Save tx data
	movlw		9									; Total of 9 bits
	movwf		iic_bit_count						;
	rrf			iic_ack_tx, W						; Shift ACK into data register
ibl													;
	rlf			iic_data_tx, F						; Get msb of data/ack
	bcf			pSDA, bSDA							;
	bsf			STATUS, RP0							;
	btfss		STATUS, C							; - Output data on SDA
	bcf			pSDA, bSDA							;
	btfsc		STATUS, C							;
	bsf			pSDA, bSDA							;
	bcf			STATUS, RP0							;
													;
	call		iic_delay							; - Delay
													;
	bsf			pSCL, bSCL							; - SCL high
													;
	call		iic_delay							; - Delay
													;
	bcf			STATUS, C							; - Input data on SDA
	btfsc		pSDA, bSDA							;
	bsf			STATUS, C							;
	rlf			iic_data_rx, F						;
	rlf			iic_ack_rx, F						;
													;
	bcf			pSCL, bSCL							; - SCL low
													;
	decfsz		iic_bit_count, F					; Decrement bit count and...
	goto		ibl									; Loop if not zero...
													;
	bsf			pSDA, bSDA							; SDA high
													;
	rrf			iic_ack_rx, F						; Fixup rx data and ACK
	rrf			iic_data_rx, F						;
	clrf		iic_ack_rx							;
	rlf			iic_ack_rx, F						;
													;
	return											; Return
													;
iic_delay											;
	movlw		5									;
	movwf		iic_dly								;
idl													;
	nop												;
	decfsz		iic_dly, F							;
	goto		idl									;
	return											;
													; **************** Serial I/O **************
													;	
SerCRLF												; - CR LF
	movlw		13									;
	call		SerTx								;
	movlw		10									;
	goto		SerTx								;
													; 
tx_hex_byte											; - Send byte as ASCII hex
	movwf		temp1								;
	swapf		temp1,W								;
	call		tx_hex_nibble						;
	movf		temp1,W								;
	;goto		tx_hex_nibble						;
													;
tx_hex_nibble										; - Send nibble as ASCII hex
	andlw		0x0F								;
	addlw		-10									;
	btfsc		STATUS,C							;
	addlw		'A'-'9'-1							;
	addlw		10+'0'								;
	; fall thru										;
													;
SerTx												;
	clrwdt											; Clear watchdog
	;bsf	asio_out_port, asio_out_bit				; Start bit
	bcf		asio_out_port, asio_out_bit				; Start bit
	movwf	asio_data								; Move W to tx data register
	movlw	9<<1 | 1								; 8 data bits + stop bit, stop bit
	movwf	asio_flags								;
	call	SerDelay								; Wait one bit time
	bcf		STATUS ,C								; Don't corrupt bit count
	rrf		asio_flags, F							; Shift stop/parity bit into carry, bit count remains
stxbitx												;
	rrf		asio_data ,F							; Shift out a bit
	btfsc	STATUS, C								; Is it a zero?
	goto	$+4										; No..
	nop												; Precise timing
	;bsf	asio_out_port, asio_out_bit				;
	bcf		asio_out_port, asio_out_bit				;
	goto	$+3										;
	;bcf	asio_out_port, asio_out_bit				;
	bsf		asio_out_port, asio_out_bit				;
	goto	$+1										;
	call	SerDelay								; Wait one bit time
	decfsz	asio_flags, F							; Dec bit count
	goto	stxbitx									; Not zero, do next bit...
	goto	SerDelay								; Wait one more bit time (2 stop bits) and return
													;
SerRx												;
	clrf	asio_data								; Clear rx char
	clrf	asio_flags								; Clear flags
srxwsbx												;
	clrwdt											; Reset watchdog
	;btfss	asio_in_port, asio_in_bit				; Wait for start bit
	btfsc	asio_in_port, asio_in_bit				; Wait for start bit
	goto	srxwsbx									;
													; - Half bit delay
	comf	asio_bit_time, W						; Get compliement of bit time
	btfsc	asio_bit_time, 1						; Add one cycle if bit 1 of bit time is set
													;  bit 0 is ignored because a one half instruction delay is not possible
	goto	$+1										;
	addlw	8										; Add 8 to W until overflow, each loop iteration takes 4 cycles
	btfss	STATUS, C								;
	goto	$-2										;
srxtsb												;
	;btfss	asio_in_port, asio_in_bit				; Check start bit
	btfsc	asio_in_port, asio_in_bit				; Check start bit
	goto	srxbadstart								; Bad start bit...
	rrf		asio_data, F							; Mark end of rx
	goto	srxsbi									; Receive the start bit...
srxbitx												;
	call	SerDelay								; Bit delay, set carry
	;btfsc	asio_in_port, asio_in_bit				; Check Rx state
	btfss	asio_in_port, asio_in_bit				; Check Rx state
	bcf		STATUS, C								; Got a zero, so clear carry
	btfsc	STATUS, C								; Update parity
	xorwf	asio_flags, F							;
	incf	asio_flags, F							; Preserve parity
srxsbi												;
	rrf		asio_data, F							; Shift in the bit
	rrf		asio_flags, F							;
	btfss	asio_flags, 5							; Done?
	goto	srxbitx									; No, do next bit...
srxbadstart											;
	rlf		asio_flags, W							; Get rx data to W, discard stop bit
	rlf		asio_data, W							;
	return											; Return
													;
													;
SerDelay											; --- Delay N + 12 cycles with one cycle granularity
													; W == 1, C == 1 upon exit
													;
	comf	asio_bit_time, W						; Get compliment of bit time to W
	bsf		STATUS, C								; Carry must be set upon exit
	btfsc	asio_bit_time, 0						; Add one cycle if bit 0 of bit time is set
	goto	$+1										;
	btfss	asio_bit_time, 1						; Add two cycles if bit 1 of bit time is set
	goto	$+3										;
	goto	$+1										;
	nop												;
	addlw	4										; Add 4 to W until overflow
	btfss	STATUS, C								;  Each loop interation takes 4 cycles
	goto	$-2										;
	retlw	1										; Set W to 1 and return
													;
													;
	end												;
													;
													;


